/**
 * Author......: See docs/credits.txt
 * License.....: MIT
 */

#include "common.h"
#include "types.h"
#include "modules.h"
#include "bitops.h"
#include "convert.h"
#include "shared.h"
#include "memory.h"

static const u32   ATTACK_EXEC    = ATTACK_EXEC_OUTSIDE_KERNEL;
static const u32   DGST_POS0      = 0;
static const u32   DGST_POS1      = 1;
static const u32   DGST_POS2      = 2;
static const u32   DGST_POS3      = 3;
static const u32   DGST_SIZE      = DGST_SIZE_4_16;
static const u32   HASH_CATEGORY  = HASH_CATEGORY_FDE;
static const char *HASH_NAME      = "LUKS v1 SHA-512 + Serpent";
static const u64   KERN_TYPE      = 14632; // old kernel used here
static const u32   OPTI_TYPE      = OPTI_TYPE_ZERO_BYTE
                                  | OPTI_TYPE_SLOW_HASH_SIMD_LOOP;
static const u64   OPTS_TYPE      = OPTS_TYPE_STOCK_MODULE
                                  | OPTS_TYPE_PT_GENERATE_LE;
static const u32   SALT_TYPE      = SALT_TYPE_EMBEDDED;
static const char *ST_PASS        = "hashcat";
static const char *ST_HASH        = "$luks$1$sha512$serpent$cbc-essiv:sha256$128$71794$f622ac496360d1a66bbf7b0842a7ea1e77696b774ce284135d312a4fddd24471$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$5135b50e847ae1cdf130491e000381faf2e658e1c11713321243edfaa4c95cd06b915b8517016038de8febd8db246de766fbd68ae3b58a435c1a0948c9b591372c57d6c9dad7ff1850da1a6ac840e6105855fd780c18b73c080d6b048cc04563769f943503b04d0376da4f8f754fde39db83c9315e4807020eb6251c1c438827784aa3acdfe315b44c8407bd9d9cf8440adeb0afc9c0c0d9fed30f92b4228f880fe31ca5b2b63e8fdb103c50a45c204cb0d90f470ef6077986b8856bfded906344c3b8f628582691b3b8ba4dadc5d091d5112528e2099e05f3bd0d86836bdc0023875b616e655f1bc04655eb695c0e98ef430d918cdf5609405ad730091019d6c4e88f55a8d701d909056f98dbda3cfd0bbe8812de0079a2c5d3ea8780f1a83deddc3018b58680d1cd631076ac4521de4a7b05c1312c3341f203f0ac765901e93d205281c3bbe951e64a1596a8a847c88e12f55d369d544d56ea0af2e7fe1d09a0380f72ef6fdd8d6be4addbf72989f623d89bbb64b9445a472cf3463de5a77ab7ebdf73024950c9539924735635b815d9dd4bd69affa0ad56255993cfeb8bb24f53a392b8a831ef12ed81a804f5cdefac2a9bcdb4f7c0ac0bd738cac0ec439baeb34a5dcdfe387696555f313a5d750ef51fa7351fc53c2b67d5eeed31826e13bf6777b051cd7cf86aab3261607e86cf129a481584d8268a059e09e1e4c19472";

u32         module_attack_exec    (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ATTACK_EXEC;     }
u32         module_dgst_pos0      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS0;       }
u32         module_dgst_pos1      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS1;       }
u32         module_dgst_pos2      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS2;       }
u32         module_dgst_pos3      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS3;       }
u32         module_dgst_size      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_SIZE;       }
u32         module_hash_category  (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_CATEGORY;   }
const char *module_hash_name      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_NAME;       }
u64         module_kern_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return KERN_TYPE;       }
u32         module_opti_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTI_TYPE;       }
u64         module_opts_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTS_TYPE;       }
u32         module_salt_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return SALT_TYPE;       }
const char *module_st_hash        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_HASH;         }
const char *module_st_pass        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_PASS;         }

#define LUKS_STRIPES        (                                   4000)
#define LUKS_SALT_LEN       (                                     32)
#define LUKS_SALT_HEX_LEN   (                      LUKS_SALT_LEN * 2)
#define LUKS_CT_LEN         (                                    512)
#define LUKS_CT_HEX_LEN     (                        LUKS_CT_LEN * 2)
#define LUKS_AF_MIN_LEN     (HC_LUKS_KEY_SIZE_128 / 8 * LUKS_STRIPES)
#define LUKS_AF_MIN_HEX_LEN (                    LUKS_AF_MIN_LEN * 2)
#define LUKS_AF_MAX_LEN     (HC_LUKS_KEY_SIZE_512 / 8 * LUKS_STRIPES)
#define LUKS_AF_MAX_HEX_LEN (                    LUKS_AF_MAX_LEN * 2)

typedef enum hc_luks_hash_type
{
  HC_LUKS_HASH_TYPE_SHA1      = 1,
  HC_LUKS_HASH_TYPE_SHA256    = 2,
  HC_LUKS_HASH_TYPE_SHA512    = 3,
  HC_LUKS_HASH_TYPE_RIPEMD160 = 4,
  HC_LUKS_HASH_TYPE_WHIRLPOOL = 5,

} hc_luks_hash_type_t;

typedef enum hc_luks_key_size
{
  HC_LUKS_KEY_SIZE_128 = 128,
  HC_LUKS_KEY_SIZE_256 = 256,
  HC_LUKS_KEY_SIZE_512 = 512,

} hc_luks_key_size_t;

typedef enum hc_luks_cipher_type
{
  HC_LUKS_CIPHER_TYPE_AES     = 1,
  HC_LUKS_CIPHER_TYPE_SERPENT = 2,
  HC_LUKS_CIPHER_TYPE_TWOFISH = 3,

} hc_luks_cipher_type_t;

typedef enum hc_luks_cipher_mode
{
  HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256 = 1,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN        = 2,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN64      = 3,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN        = 4,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN64      = 5,

} hc_luks_cipher_mode_t;

typedef struct luks
{
  int hash_type;   // hc_luks_hash_type_t
  int key_size;    // hc_luks_key_size_t
  int cipher_type; // hc_luks_cipher_type_t
  int cipher_mode; // hc_luks_cipher_mode_t

  u32 ct_buf[LUKS_CT_LEN / 4];

  u32 af_buf[LUKS_AF_MAX_LEN / 4];
  u32 af_len;

} luks_t;

typedef struct luks_tmp
{
  u32 ipad32[8];
  u64 ipad64[8];

  u32 opad32[8];
  u64 opad64[8];

  u32 dgst32[32];
  u64 dgst64[16];

  u32 out32[32];
  u64 out64[16];

} luks_tmp_t;

static const char *SIGNATURE_LUKS = "$luks$1$sha512$serpent$";

bool module_unstable_warning (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra, MAYBE_UNUSED const hc_device_param_t *device_param)
{
  if ((device_param->opencl_platform_vendor_id == VENDOR_ID_APPLE) && (device_param->opencl_device_type & CL_DEVICE_TYPE_GPU))
  {
    if (device_param->is_metal == true)
    {
      if (strncmp (device_param->device_name, "Intel", 5) == 0)
      {
        // Intel Iris Graphics, Metal Version 244.303: failed to create 'm14632_init' pipeline, timeout reached
        return true;
      }
    }
  }

  return false;
}

u64 module_esalt_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 esalt_size = (const u64) sizeof (luks_t);

  return esalt_size;
}

u64 module_tmp_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 tmp_size = (const u64) sizeof (luks_tmp_t);

  return tmp_size;
}

int module_hash_decode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED void *digest_buf, MAYBE_UNUSED salt_t *salt, MAYBE_UNUSED void *esalt_buf, MAYBE_UNUSED void *hook_salt_buf, MAYBE_UNUSED hashinfo_t *hash_info, const char *line_buf, MAYBE_UNUSED const int line_len)
{
  luks_t *luks = (luks_t *) esalt_buf;

  hc_token_t token;

  memset (&token, 0, sizeof (hc_token_t));

  token.token_cnt  = 7;

  token.signatures_cnt    = 1;
  token.signatures_buf[0] = SIGNATURE_LUKS;

  // signature with hash and cipher type
  token.len[0]     = 23;
  token.attr[0]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_SIGNATURE;

  // cipher mode
  token.sep[1]     = '$';
  token.len_min[1] = 9;
  token.len_max[1] = 16;
  token.attr[1]    = TOKEN_ATTR_VERIFY_LENGTH;

  // key size
  token.sep[2]     = '$';
  token.len[2]     = 3;
  token.attr[2]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // iter
  token.sep[3]     = '$';
  token.len_min[3] = 1;
  token.len_max[3] = 10;
  token.attr[3]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // salt
  token.sep[4]     = '$';
  token.len[4]     = LUKS_SALT_HEX_LEN;
  token.attr[4]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // af
  token.sep[5]     = '$';
  token.len_min[5] = LUKS_AF_MIN_HEX_LEN;
  token.len_max[5] = LUKS_AF_MAX_HEX_LEN;
  token.attr[5]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // ct
  token.sep[6]     = '$';
  token.len[6]     = LUKS_CT_HEX_LEN;
  token.attr[6]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  const int rc_tokenizer = input_tokenizer ((const u8 *) line_buf, line_len, &token);

  if (rc_tokenizer != PARSER_OK) return (rc_tokenizer);

  // hash type

  luks->hash_type = HC_LUKS_HASH_TYPE_SHA512;

  // cipher type

  luks->cipher_type = HC_LUKS_CIPHER_TYPE_SERPENT;

  // cipher mode

  const u8 *cipher_mode_pos = token.buf[1];
  const u32 cipher_mode_len = token.len[1];

  if ((strncmp ((const char *) cipher_mode_pos, "cbc-essiv:sha256", 16) == 0) && (cipher_mode_len == 16))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN64;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN64;
  }
  else
  {
    return (PARSER_LUKS_CIPHER_MODE);
  }

  // key size

  const u8 *key_size_pos = token.buf[2];

  const u32 key_size = hc_strtoul ((const char *) key_size_pos, NULL, 10);

  if (key_size == 128)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_128;
  }
  else if (key_size == 256)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_256;
  }
  else if (key_size == 512)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_512;
  }
  else
  {
    return (PARSER_LUKS_KEY_SIZE);
  }

  // salt

  const u8 *salt_pos = token.buf[4];

  salt->salt_len = hex_decode (salt_pos, LUKS_SALT_HEX_LEN, (u8 *) salt->salt_buf);

  // iter

  const u8 *iter_pos = token.buf[3];

  const u32 iter = hc_strtoul ((const char *) iter_pos, NULL, 10);

  salt->salt_iter = iter - 1;

  // af

  u32 af_len = token.len[5] / 2;

  if (af_len != (key_size / 8) * LUKS_STRIPES) return (PARSER_LUKS_FILE_SIZE);

  const u8 *af_pos = token.buf[5];

  luks->af_len = hex_decode (af_pos, token.len[5], (u8 *) luks->af_buf);

  // ct

  u32 ct_len = token.len[6] / 2;

  if (ct_len != LUKS_CT_LEN) return (PARSER_LUKS_FILE_SIZE);

  const u8 *ct_pos = token.buf[6];

  hex_decode (ct_pos, LUKS_CT_HEX_LEN, (u8 *) luks->ct_buf);

  return (PARSER_OK);
}

int module_hash_encode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const void *digest_buf, MAYBE_UNUSED const salt_t *salt, MAYBE_UNUSED const void *esalt_buf, MAYBE_UNUSED const void *hook_salt_buf, MAYBE_UNUSED const hashinfo_t *hash_info, char *line_buf, MAYBE_UNUSED const int line_size)
{
  const luks_t *luks = (const luks_t *) esalt_buf;

  // cipher mode

  const char *cipher_mode = "";

  switch (luks->cipher_mode)
  {
    case HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256: cipher_mode = "cbc-essiv:sha256"; break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN:        cipher_mode = "cbc-plain";        break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN64:      cipher_mode = "cbc-plain64";      break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN:        cipher_mode = "xts-plain";        break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN64:      cipher_mode = "xts-plain64";      break;
    default:                                                                     return 0;
  }

  // key size

  u32 key_size = 0;

  switch (luks->key_size)
  {
    case HC_LUKS_KEY_SIZE_128: key_size = 128; break;
    case HC_LUKS_KEY_SIZE_256: key_size = 256; break;
    case HC_LUKS_KEY_SIZE_512: key_size = 512; break;
    default:                                   return 0;
  }

  // iterations

  u32 iterations = salt->salt_iter + 1;

  // salt

  char salt_buf[LUKS_SALT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) salt->salt_buf, LUKS_SALT_LEN, (u8 *) salt_buf);

  // af

  char af_buf[LUKS_AF_MAX_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->af_buf, luks->af_len, (u8 *) af_buf);

  // ct

  char ct_buf[LUKS_CT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->ct_buf, LUKS_CT_LEN, (u8 *) ct_buf);

  // output

  int line_len = snprintf (line_buf, line_size, "%s%s$%u$%u$%s$%s$%s",
    SIGNATURE_LUKS,
    cipher_mode,
    key_size,
    iterations,
    salt_buf,
    af_buf,
    ct_buf);

  return line_len;
}

void module_init (module_ctx_t *module_ctx)
{
  module_ctx->module_context_size             = MODULE_CONTEXT_SIZE_CURRENT;
  module_ctx->module_interface_version        = MODULE_INTERFACE_VERSION_CURRENT;

  module_ctx->module_attack_exec              = module_attack_exec;
  module_ctx->module_benchmark_esalt          = MODULE_DEFAULT;
  module_ctx->module_benchmark_hook_salt      = MODULE_DEFAULT;
  module_ctx->module_benchmark_mask           = MODULE_DEFAULT;
  module_ctx->module_benchmark_charset        = MODULE_DEFAULT;
  module_ctx->module_benchmark_salt           = MODULE_DEFAULT;
  module_ctx->module_bridge_name              = MODULE_DEFAULT;
  module_ctx->module_bridge_type              = MODULE_DEFAULT;
  module_ctx->module_build_plain_postprocess  = MODULE_DEFAULT;
  module_ctx->module_deep_comp_kernel         = MODULE_DEFAULT;
  module_ctx->module_deprecated_notice        = MODULE_DEFAULT;
  module_ctx->module_dgst_pos0                = module_dgst_pos0;
  module_ctx->module_dgst_pos1                = module_dgst_pos1;
  module_ctx->module_dgst_pos2                = module_dgst_pos2;
  module_ctx->module_dgst_pos3                = module_dgst_pos3;
  module_ctx->module_dgst_size                = module_dgst_size;
  module_ctx->module_dictstat_disable         = MODULE_DEFAULT;
  module_ctx->module_esalt_size               = module_esalt_size;
  module_ctx->module_extra_buffer_size        = MODULE_DEFAULT;
  module_ctx->module_extra_tmp_size           = MODULE_DEFAULT;
  module_ctx->module_extra_tuningdb_block     = MODULE_DEFAULT;
  module_ctx->module_forced_outfile_format    = MODULE_DEFAULT;
  module_ctx->module_hash_binary_count        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_parse        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_save         = MODULE_DEFAULT;
  module_ctx->module_hash_decode_postprocess  = MODULE_DEFAULT;
  module_ctx->module_hash_decode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_decode_zero_hash    = MODULE_DEFAULT;
  module_ctx->module_hash_decode              = module_hash_decode;
  module_ctx->module_hash_encode_status       = MODULE_DEFAULT;
  module_ctx->module_hash_encode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_encode              = module_hash_encode;
  module_ctx->module_hash_init_selftest       = MODULE_DEFAULT;
  module_ctx->module_hash_mode                = MODULE_DEFAULT;
  module_ctx->module_hash_category            = module_hash_category;
  module_ctx->module_hash_name                = module_hash_name;
  module_ctx->module_hashes_count_min         = MODULE_DEFAULT;
  module_ctx->module_hashes_count_max         = MODULE_DEFAULT;
  module_ctx->module_hlfmt_disable            = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_size    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_init    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_term    = MODULE_DEFAULT;
  module_ctx->module_hook12                   = MODULE_DEFAULT;
  module_ctx->module_hook23                   = MODULE_DEFAULT;
  module_ctx->module_hook_salt_size           = MODULE_DEFAULT;
  module_ctx->module_hook_size                = MODULE_DEFAULT;
  module_ctx->module_jit_build_options        = MODULE_DEFAULT;
  module_ctx->module_jit_cache_disable        = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_max       = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_min       = MODULE_DEFAULT;
  module_ctx->module_kern_type                = module_kern_type;
  module_ctx->module_kern_type_dynamic        = MODULE_DEFAULT;
  module_ctx->module_opti_type                = module_opti_type;
  module_ctx->module_opts_type                = module_opts_type;
  module_ctx->module_outfile_check_disable    = MODULE_DEFAULT;
  module_ctx->module_outfile_check_nocomp     = MODULE_DEFAULT;
  module_ctx->module_potfile_custom_check     = MODULE_DEFAULT;
  module_ctx->module_potfile_disable          = MODULE_DEFAULT;
  module_ctx->module_potfile_keep_all_hashes  = MODULE_DEFAULT;
  module_ctx->module_pwdump_column            = MODULE_DEFAULT;
  module_ctx->module_pw_max                   = MODULE_DEFAULT;
  module_ctx->module_pw_min                   = MODULE_DEFAULT;
  module_ctx->module_salt_max                 = MODULE_DEFAULT;
  module_ctx->module_salt_min                 = MODULE_DEFAULT;
  module_ctx->module_salt_type                = module_salt_type;
  module_ctx->module_separator                = MODULE_DEFAULT;
  module_ctx->module_st_hash                  = module_st_hash;
  module_ctx->module_st_pass                  = module_st_pass;
  module_ctx->module_tmp_size                 = module_tmp_size;
  module_ctx->module_unstable_warning         = module_unstable_warning;
  module_ctx->module_warmup_disable           = MODULE_DEFAULT;
}
